# @file slhdsa_makesig_test.py 
# @version 1.1.0 (2026-02-15T08:23Z)
# @author David Ireland <https://di-mgt.com.au/contact>
# @copyright 2023-26 DI Management Services Pty Ltd
# @license Apache-2.0

"""Tests for slhdsa_makesig"""

import os
from slhdsa_makesig import slhdsa_sign_internal, slhdsa_sign, params_dict, show_params

TESTDIR = "./tests/"


def save_sig_to_file(algname, sighex, sigannotated, dirpath):
    # Write sig to a file in tests subfolder
    fname = dirpath + "sig-" + algname + "-out.txt"
    with open(fname, 'w') as f: f.write(sigannotated)
    print("Created file", fname)

    fname = fname + ".raw.txt"
    with open(fname, 'w') as f: f.write(sighex)
    print("Created file", fname)


def headtail(s, headlen):
    """Display a long string split between head and tail"""
    if len(s) <= 2 * headlen:
        return s
    outstr = s[:headlen] + "\n..." + s[-headlen:]
    return outstr


############
# TESTS
############
# All tests from
    # https://github.com/usnistgov/ACVP-Server/blob/master/gen-val/json-files/SLH-DSA-sigGen-FIPS205/internalProjection.json
    # RELEASE/v1.1.0.38

def case_sha2_128f_1():
    # INPUT FOR TEST CASE tcId:1 SHA2-128f using sign_internal
    test_title = 'tcId: 1'
    algname = "SLH-DSA-SHA2-128f"
    print(f"{algname} {test_title}")
    params = params_dict[algname]
    show_params(algname)
    sk = 'D5213BA4BB6470F1B9EDA88CBC94E6277A58A951EF7F2B81461DBAC41B5A6B83FA495FB834DEFEA7CC96A81309479135A67029E90668C5A58B96E60111491F3D'
    pk = 'FA495FB834DEFEA7CC96A81309479135A67029E90668C5A58B96E60111491F3D'
    msg = \
        '00003F'
    addrnd = ''
    # Expected hash of signature
    expected_hash = 'd3537e05ae63876bf78015ff86cc9e284f91cabfac191298fbaa375596351d55'

    # Call with expected intermediate values
    sighex, sigannotated = slhdsa_sign_internal(sk, msg, addrnd, params, 
        Rok='bd40e6d66893f38d5c5fad99e4885329',
        Hmsgok='cafc639d5e550807f84bae9f7eb8da2077cd579fd484522072fcdcef4b8fdb03b028',
        fors_sig_sk_0='925bb207d49e62bcb9b1c4685154a8b3',
        fors_pk_ok='33af163817cd6c2bea881ddf7d2b89ab',
        sig_hash_ok='d3537e05ae63876bf78015ff86cc9e284f91cabfac191298fbaa375596351d55')
    save_sig_to_file(params.name, sighex, sigannotated, TESTDIR)


def case_sha2_128s_163():
    # RELEASE/v1.1.0.38
    test_title = 'tcId: 163'
    algname = "SLH-DSA-SHA2-128s"
    params = params_dict[algname]
    show_params(algname)
    sk = '4678D1F07C682516F24FDF63AE47241BE4ACDA3EF56150C458C6BC477F14E75EA8B8365A9FC2A6877D2A6237C687AB41E38F37E0274FFFBD77655A1DB6C446C1'
    msg = \
        '1B'
    addrnd = ''
    context = 'CAA7B50B2763D195BFA1E5793E17A7CD5DFBD8A163BF6D876CEC512CFC97AA9D1D76E7700CBACD1F2D8371766FDFAE3CB0EA'
    # Expected hash of signature
    expected_hash = '0d6dc439e181e2a50ecdc4d6e9dcd7a0644feafebc0e6138a78a91cb748bb22c'

    sighex, sigannotated = slhdsa_sign(params, msg, sk, addrnd, ctx=context)
    save_sig_to_file(params.name, sighex, sigannotated, TESTDIR)
    print(f"HASH(ok) ={expected_hash}")
    print(headtail(sighex, 48*3))


def case_sha2_192s_280():
    # RELEASE/v1.1.0.38
    test_title = 'tcId: 280'
    algname = "SLH-DSA-SHA2-192s"
    params = params_dict[algname]
    show_params(algname)
    sk = 'D43B5E671993E1C61A37B2BDACDBE2FAA4AF4BF3CECBBAABD5D88E3E6B9DF115ADD66703EA44E882D0F9610D1F9EB5915E174C6ED52A0AE8989790AD3D3CF67F4D84C4D596BA15106651D49F332ED8EC1A8CA745D961EE3EFC441CBC71ECEC94'
    msg = \
        '6C7BE9B864591107F6087F82ED9D60E1D8D8700D0081567B7B8296A41B178FA0D7B838AF85423B9E2C54CF8249E2A1C73D420F88103FB0A42B6EFBA0F83D5BC58AC8658E649D24E860814CB12581EA76238FE276DD19693C479D4DAA2A9B1F7905F4033359C45620B9ECA1797CF7417918253C18B760BA8EFCBE53BBBEDED502FAC8A867FA029C5FA79AD797A765FB9A580B88FF9E1A42AD5CFE36F623FDBA908449DF77929984A025DA32F90E168C32B576B8786E594CD9403B017FA3C239D994E5AE3498B1111184F759EAF483E95D7ED9B85DD74E76327D7728A3AA09977FC5267CF7740A0B6ED490C7E48CC95169DFAC893A41FE0010DD223FA41FE5A14CDCE6B445ECC8090714FA5CD75F5354BD1C48894C557365796C69BF7D12206830682737E1876DB168F8DEAEA7786D7A9C9F51C48816097313C73BB39E3001E6C2A1D445D9EFF40EE0D09F55C8872CB52008F07FF8E20A05AD0027D5964E82074DA736D32D8E828D1F4AA7921BB7743B9DD10221384F8DF87502D35C7D13598762BA0C3CC3CA86222928CF86DB7F07815DA5E9EFD6E5C25AE4F99DB4B1ABB6EED082112238D1E187C0D806FAAA72F213B684083302290A74DC440EA45147FB2F237AF8CE92BD0565D8EC18E1599F9FF29EEFFBACDBD969C851A4DF3D36B973D9BA5F9949659102B131E30C6BF25C19BC82455EC420B58FBDC45D5F956F3648E5580A3D725C2138FCB31C626A962D26B1394F747F9ECCE3F7D2F56B9D8EB6CE944DA399180D53'
    addrnd = ''
    context = ''
    # Expected hash of signature
    expected_hash = '5ddd7f5972c3bfbf6dd0fc42dbe4398216192a34a62c4d4bd23b075073ae07c5'

    sighex, sigannotated = slhdsa_sign(params, msg, sk, internal=True)
    save_sig_to_file(params.name, sighex, sigannotated, TESTDIR)
    print(f"HASH(ok) ={expected_hash}")
    print(headtail(sighex, 48*3))


def case_sha2_256s_513():
    # RELEASE/v1.1.0.38
    test_title = 'tcId: 513'
    algname = "SLH-DSA-SHA2-256s"
    params = params_dict[algname]
    show_params(algname)
    sk = 'EA6C803A4879049115A18CEA44F93B3CAC7E9A0335821336BEBEC62A328E7146B684754C25A078CFB90AF0CEB5A945F2CAE633AC00C576F6D12873559646F2C2818403D3DDC68CE8D21CB6D7F456AF1E5E69A1C6744E2712C856E853107397A79B2E8E160B71B6AA50553FCFD182F2693D34E1D9B18EB03852931995B7491EA2'
    msg = \
        '67'
    addrnd = 'A1FDBFBE02E0DA2CA5252DB2B8F715D852CE5B815B929C99D95CDD4C446080DB'
    context = ''
    # Expected hash of signature
    expected_hash = '52cf9084167f5948b337bc2c498462492f8d3945b3d7e46486dc1ca2607626da'

    sighex, sigannotated = slhdsa_sign(params, msg, sk, addrnd)
    save_sig_to_file(params.name, sighex, sigannotated, TESTDIR)
    print(f"HASH(ok) ={expected_hash}")
    print(headtail(sighex, 48*3))


def case_sha2_192f_50():
    # RELEASE/v1.1.0.38
    test_title = 'tcId: 50'
    algname = "SLH-DSA-SHA2-192f"
    params = params_dict[algname]
    show_params(algname)
    sk = 'AC08F51ECC17357D539C77C8C50C32D0790BBB84772C8EAA75EAA850FC369768BE6BD6685C7F62AE1FA943886450DAC0A6EA3FF86516ED898CF0A66BAAB78086B45DDC4FF77946BEFEB9AEE2C0FC1EB963A5F6B738DB52B965BE6F4F897F659B'
    pk = 'A6EA3FF86516ED898CF0A66BAAB78086B45DDC4FF77946BEFEB9AEE2C0FC1EB963A5F6B738DB52B965BE6F4F897F659B'
    msg = \
        'FFC2FCCDBD0D4C4400F4BF36155EE74FAB9647B077CFC17141DAA95C8F5FAD5882A1D3FD06F6F3D84CE6F67BD508A9EA52FFE5FD938086D2C002C3B0E5992BF66F943EBF493A4E9093DB31FEA18173058B8EF6BA561DEDC1601C38548687B37EE497FC490D357949B93B1B8142E7EFF8574A'
    addrnd = '994B8B7416B64A5D909C6E27B963C6CE41F66F85503BFEE5'
    # Expected hash of signature
    expected_hash = '9af0a8d67443296282559b3af12b86128b0f67f41b873cc91f0f6271bd10059a'

    sighex, sigannotated = slhdsa_sign(params, msg, sk, addrnd, internal=True)
    save_sig_to_file(params.name, sighex, sigannotated, TESTDIR)
    print(f"HASH(ok) ={expected_hash}")
    print(headtail(sighex, 48*3))


def case_sha2_256f_353():
    # RELEASE/v1.1.0.38
    test_title = 'tcId: 353'
    algname = "SLH-DSA-SHA2-256f"
    params = params_dict[algname]
    show_params(algname)
    sk = '760A91EB234223F00F31DEB8E82ABDE1CEE2ACDE801E20B2D363A58AC9B1936021E8406266D9A262820B549DC0785588C3B39E02A6DB844CB885332FE89420C70C551E5A8D85111472D94A18959A7E70510C8A02339B7238AD9EFA18BDB8A716E9B59484DC90BF96C04BB2FE09C831EC5439BB0BEE83D3FAF1C67E7D39FB9EA7'
    msg = \
        'F4'
    addrnd = '1E0AAE0652400E67C1B778669407F58C462B74DE099802FDAA23B59DACE50BE8'
    # Expected hash of signature
    expected_hash = '6f51aed4cf11a9794dcbbf4a5821d5b19f0c2e13515cb3862017bf2460fe1910'

    sighex, sigannotated = slhdsa_sign(params, msg, sk, addrnd)
    save_sig_to_file(params.name, sighex, sigannotated, TESTDIR)
    print(f"HASH(ok) ={expected_hash}")
    print(headtail(sighex, 48*3))


def case_shake_256f_99():
    # RELEASE/v1.1.0.38
    test_title = 'tcId: 99'
    algname = "SLH-DSA-SHAKE-256f"
    params = params_dict[algname]
    show_params(algname)
    sk = 'B6BD6EE787A6C4FB5D3FC4A39EE0E1BD6D873E7CBC10F0FBAA3DBC4348A756B48CC2B6167F4D4A17F019278D7F9FF02BB859C5504290BB555B4CD800C7BCEBE19F45CC8BB6B0CCD1BA293ECFF65BEE6A360DFC34E8488495E46DBF8190DA59333A09CA93B04D715905D5C65A9133437D8AA816CF50ED7F725AC45671E686646A'
    pk = '9F45CC8BB6B0CCD1BA293ECFF65BEE6A360DFC34E8488495E46DBF8190DA59333A09CA93B04D715905D5C65A9133437D8AA816CF50ED7F725AC45671E686646A'
    msg = \
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
    context = '42491ABE1211CCC14E1D2CFC53B49DB88B93BE8BE2F93CC6A12586E5807C17D66F7C95D5C1BF6BA1F463E80D13C4CAED8CB534C5629380D71EDB7D'
    addrnd = ''
    # Expected hash of signature
    expected_hash = '8277038297d2c49c3da6bad2548214380926300d0c36a4dc065d5a932351e081'

    sighex, sigannotated = slhdsa_sign(params, msg, sk, ctx=context)
    save_sig_to_file(params.name, sighex, sigannotated, TESTDIR)
    print(f"HASH(ok) ={expected_hash}")
    print(headtail(sighex, 48*3))


def case_shake_256s_564():
    # RELEASE/v1.1.0.38
    test_title = 'tcId: 564'
    algname = "SLH-DSA-SHAKE-256s"
    params = params_dict[algname]
    show_params(algname)
    sk = 'C3EAB12E81E91DE43F0699449CD5BBD03FD1D9D96EF78838D3F25A879B2453F386A5707169B7E54C598F0D4D0E46266687A06515C9B9A6CA453E6EA3A1C976FA59D4D4B7BD5470E6EC76DB6B081A627D7E293CE09ED1907C619C30106273339E0BBE101A1097D2756382E67D28B3A2C2F527363408670F2883F3A1D4F75B4104'
    msg = \
        '2E'
    context = 'AB6639A2EDD137D085D12D810FBCFE5B18A5B9A4071A4CB3904C505BD7B283611E4B28F4F5F9F88F95609E313D9E729D4C1AF054853527D45B'
    addrnd = '7AEE5F7E23072CB2339D3B6BF699E053D6F15883E5890D254DF1423ED12416DC'
    # Expected hash of signature
    expected_hash = '865a9b338b9534d37dc2a6175407cc9bb3f699ebf782d43bd799dac379f27099'

    sighex, sigannotated = slhdsa_sign(params, msg, sk, addrnd, ctx=context)
    save_sig_to_file(params.name, sighex, sigannotated, TESTDIR)
    print(f"HASH(ok) ={expected_hash}")
    print(headtail(sighex, 48*3))


def case_shake_128f_62():
    # RELEASE/v1.1.0.38
    test_title = 'tcId: 62'
    algname = "SLH-DSA-SHAKE-128f"
    params = params_dict[algname]
    show_params(algname)
    sk = 'FA16A525899CD4142085D85ADA1D26088A93857E49F26E97270245CFFB65A1A7957D8E7F0EB215347708AFDCBB6BDCA84385726F923BB8AB3E1A6C78250EE39C'
    msg = \
        'B461D1F7D7D6932BE0C200B7303373C6F47B87B2B7C21C3EED'
    context = '6624EE1939FC2198CB503E18778BD6B5D3A60F8AF9493EB0485448B6CA08507C4209EEFA37C783FB64074BA24DDEFD3CEA251675B46B2F551E25E0BCF7869F16F38BF775C8C1DAE91BC4F5289E5DD629113F3D6A1F4307397EB7BB12B09FBF5A2FE90023070E7C288F35B9BCA5AE0CDA37915CB6E0CF00F3AFFF8344E96A1FD5809BD6AA370DBB6DF61E391F37B73902FCA57CD07440F606933831C37C7246C954F073DFB0087F'
    addrnd = ''
    # Expected hash of signature
    expected_hash = '9bda3d12f36d41315b28989a950f708d57565adf3670cc714307aa8df66b6bb9'

    sighex, sigannotated = slhdsa_sign(params, msg, sk, ctx=context)
    save_sig_to_file(params.name, sighex, sigannotated, TESTDIR)
    print(f"HASH(ok) ={expected_hash}")
    print(headtail(sighex, 48*3))


def case_shake_192f_395():
    # RELEASE/v1.1.0.38
    test_title = 'tcId: 395'
    algname = "SLH-DSA-SHAKE-192f"
    params = params_dict[algname]
    show_params(algname)
    sk = '441CCD992DBE29943E15652BA12AAD1049DD78606B77D1DE16A0D1386634FC7A825EEA88E0D6C4EE3FBCF89330221111AAB5358F67B2EAF117B6AC554E5A33B5115C35242EE9729AD1C930453BFE54B7BF20A6549CF38D0230281409CB6CAED7'
    msg = \
        '4AD6C5700DDADEA081E794E566FAE0A97103EFCBAF8BE57485D5BDEABF4F31532F943694E7FC7E38BEAA3EE0357E082BD340362C6591DA806AAD6C8C139C4AC0C7C9636F26B4EAA830555C890C4F21F262D0870910F91C92035C48D1A4AF8E1B334B3057706AFB5B4A47268223DEDDB37F113BA1B77E0BF720D4D9220F34EF56D88068E902A0049D723E881BF13D67DD7E58069405669B9C3433F774086E99D36217C08ABC5C0B85A64008531A402B1849E190A907526E980B24450E26AADE7A91A35B1C55B0BF0F020EE773FFA7C17A411E1FE6E92B76BE34163B7B6B0D948DB408FADCBA6C3668C879D57E11A2AF47A0B11CBA3792838C7F16898C198788D78CF815AEEE612B5ED856796CB8F5D5BDA25403F3E52085CAB7E93DE6CB0EBF42E68D73E65F519BA3A502DBBC5D3A75BCFD549BD8C96E6400DE74008A04D060B413D6A72EB2D2113D2CCD624A21633EB77342B72BC2D2188DBB25B8708EC870074CFC209BF2E9FEB4805DAAE2FC202596E58E5275D8ACB033729A17B086919BD7AF5E6A46FA2446C5B78FD873068B27D2D63F02299F2066B6431AC038AEB64458D1B16D23F3FBD5E7B8F1953E63F34E33D2C566CBE96C881D825D42D00CFEB64D'
    context = '8377F63B64F0B0C71E5600442DD21E00D65497C45FDB5702229731D384B74A1829BD65A25D2B552EAEA0518F1DE136413E00EAB4333CBBE9195367BA1F5F6770E51D2663135D00C008CA001BC5715689525032EB272F364F2079FF5DEC5DF9AFBB1600736B889171EB12E3698CDC6219B15317180E2387E59192EBA5CEECD2EC348CF7278A07DD6969410FAE15C8E20C1ECE4ABF4A2D3BB33E8A7B48F2FAB2FBC1F8326EE981E7FB126146'
    addrnd = '18E49093835C550197EA4075371BBF9E0F6B7B326A1A12FE'
    # Expected hash of signature
    expected_hash = 'b436b2f6bffb0cc00c5fbfa46a7eb32ac80d99bf25bd39439186b44c6e4d5e97'

    sighex, sigannotated = slhdsa_sign(params, msg, sk, ctx=context, addrnd=addrnd)
    save_sig_to_file(params.name, sighex, sigannotated, TESTDIR)
    print(f"HASH(ok) ={expected_hash}")
    print(headtail(sighex, 48*3))


def case_shake_256f_395():
    # RELEASE/v1.1.0.38
    test_title = 'tcId: 395'
    algname = "SLH-DSA-SHAKE-256f"
    params = params_dict[algname]
    show_params(algname)
    sk = '441CCD992DBE29943E15652BA12AAD1049DD78606B77D1DE16A0D1386634FC7A825EEA88E0D6C4EE3FBCF89330221111AAB5358F67B2EAF117B6AC554E5A33B5115C35242EE9729AD1C930453BFE54B7BF20A6549CF38D0230281409CB6CAED7'
    msg = \
        '4AD6C5700DDADEA081E794E566FAE0A97103EFCBAF8BE57485D5BDEABF4F31532F943694E7FC7E38BEAA3EE0357E082BD340362C6591DA806AAD6C8C139C4AC0C7C9636F26B4EAA830555C890C4F21F262D0870910F91C92035C48D1A4AF8E1B334B3057706AFB5B4A47268223DEDDB37F113BA1B77E0BF720D4D9220F34EF56D88068E902A0049D723E881BF13D67DD7E58069405669B9C3433F774086E99D36217C08ABC5C0B85A64008531A402B1849E190A907526E980B24450E26AADE7A91A35B1C55B0BF0F020EE773FFA7C17A411E1FE6E92B76BE34163B7B6B0D948DB408FADCBA6C3668C879D57E11A2AF47A0B11CBA3792838C7F16898C198788D78CF815AEEE612B5ED856796CB8F5D5BDA25403F3E52085CAB7E93DE6CB0EBF42E68D73E65F519BA3A502DBBC5D3A75BCFD549BD8C96E6400DE74008A04D060B413D6A72EB2D2113D2CCD624A21633EB77342B72BC2D2188DBB25B8708EC870074CFC209BF2E9FEB4805DAAE2FC202596E58E5275D8ACB033729A17B086919BD7AF5E6A46FA2446C5B78FD873068B27D2D63F02299F2066B6431AC038AEB64458D1B16D23F3FBD5E7B8F1953E63F34E33D2C566CBE96C881D825D42D00CFEB64D'
    context = '8377F63B64F0B0C71E5600442DD21E00D65497C45FDB5702229731D384B74A1829BD65A25D2B552EAEA0518F1DE136413E00EAB4333CBBE9195367BA1F5F6770E51D2663135D00C008CA001BC5715689525032EB272F364F2079FF5DEC5DF9AFBB1600736B889171EB12E3698CDC6219B15317180E2387E59192EBA5CEECD2EC348CF7278A07DD6969410FAE15C8E20C1ECE4ABF4A2D3BB33E8A7B48F2FAB2FBC1F8326EE981E7FB126146'
    addrnd = '18E49093835C550197EA4075371BBF9E0F6B7B326A1A12FE'
    # Expected hash of signature
    expected_hash = 'b436b2f6bffb0cc00c5fbfa46a7eb32ac80d99bf25bd39439186b44c6e4d5e97'

    sighex, sigannotated = slhdsa_sign(params, msg, sk, ctx=context, addrnd=addrnd)
    save_sig_to_file(params.name, sighex, sigannotated, TESTDIR)
    print(f"HASH(ok) ={expected_hash}")
    print(headtail(sighex, 48*3))


def case_shake_128s_219():
    # RELEASE/v1.1.0.38
    test_title = 'tcId: 219'
    algname = "SLH-DSA-SHAKE-128s"
    params = params_dict[algname]
    show_params(algname)
    sk = '2F472F93A618BDAD05C7810D3A7F8F63D2BAAF696A99A8BFD217CA4034C134F4F651B527ADCE4768E8DEF61D0127067AAAA10EF1ADA127F0F3023F0EE288263D'
    msg = \
        '3F2ED82C5FC3C781A087C54A2FB7A11D303D825AD952CBBDD24839C67771BEF856CC171315B0EABB3BC51AFDF5A2AA9EFE7642926250935641EDDC4A7E0247BB84BAEDD955A09473D0E4AB6DA90DAAAF92B7C9BF2C43B343A66CB3A923BB9FE632C30A9F523D56C7356ED5159439C3F59700061EE87A7F8F5A3F16AB68C9E2A6FA3FA66C0341AF99D5B239ABD30D83BD181B7305619A25D9300EE102D62B7A6F1AA92217F6DC100A0567B80E53AF9D3B41EDD7E238FC4F1C579D48144BB5008EC08D5B7049C4AAF0FDAC02B0B67C7679222DB7D6BA3AFC75196BDCA0CC0EB608186F8BCA29BF69F9C3911C35716988333D152202BD19FD4B578430B2BD6E821AE174193FB0B48E442F75EFB78654FA4A1E9A0E4E29977C32AE3B7FB804996EB3'
    context = ''
    addrnd = ''
    # Expected hash of signature
    expected_hash = '9646c4858c06fa694bf151cc423380838f862f34ad7321a718420a9de030bf58'

    sighex, sigannotated = slhdsa_sign(params, msg, sk)
    save_sig_to_file(params.name, sighex, sigannotated, TESTDIR)
    print(f"HASH(ok) ={expected_hash}")
    print(headtail(sighex, 48*3))


def case_shake_192s_546():
    # RELEASE/v1.1.0.38
    test_title = 'tcId: 546'
    algname = "SLH-DSA-SHAKE-192s"
    params = params_dict[algname]
    show_params(algname)
    sk = '155A7A297E599A27E21A15F3C794E4FE1F686B51FB373EA82C00E02754A81B37393229703B11FAFEB3657BB2B91649C4FE72DE04F9C8E07E826CAF55073621E32DB4136BDCE72C346CD707C5C9E77D6676A284092F602F399759983DA9267F4D'
    msg = \
        'DFFE144D01A01227BF102981E0B6CD15CB09906AA96BEFECD325D750ACD3B3A159B522E728A0BA532E70EE497917D289C116434FEBE83302E63CEDBBF34097AD59A398B33F8F2C0F587B53DD6ED529C1E1CA9143810C44F268627754AF13CE814D5D1FA2D8B4545359C55C30DB9C645D919DCFC1C2E40B7946716793E386D4B5AB2E7AD332BA5EB08C54288C868B63D0F55ADA84C87AB0C730515D3765E0A78251E2BB3E6D5AB76B21973C72AC3BCBA3A9CD748A082697676DE72DEA268048C8C7D5C8512E0C9D6AE5E3AA5FE11DAF5E3B83B7ADA96E571A33EA4B0BC59C4FCCF71D905C990626617A1EF96040CA51B9F2F286754C40E657316158556D1888DE43EB2A32D6A251C2F804EE8FC6887B8B3FB9234D7B93B7224F66FEEAC91E6568AEBFE9BC56907216400108046235BD304C2174D381369E2CA780C349B991C3010950C584B9C8ACB38DA58319BF5FA312AFB77C8D206FB97343A2840244FFBCC41BC86305486DE7C9AE94423BAAE2CE2C8307487555DFC96C560EF1A0915B47CFE02DFC8E8B261AF82CD26CE9DB10813542165E8BCF9CF21757EE2C6266149B1A7D1E4B2C181142E193181148F6205B837058B52D75C5EE2C5A14A0CD08AD113E1A630A3793F28EF843D351BEC94C2CF7BC350F7753B67D8856DF1EA60986B5E4D6DB94231817B980CF08E3F24D440B79BBAA53BE92E719AF86224F4C735E0027C69F02D4C5E7A10FF3A15F6D6503AAA52C92040AAD7B84D9AF590BB927AC222F7549F6F8EA380BBFB4013916C1C90891DC76F9968D7671BD98ED0DD1A756102DB820366E95871C262991801E381CC6BC2AB21037DADB4FACB78C80065E35513520723288863344ECAED471DD22294918F46B06F98B30A0EBA8BDA2D973CAFA9B6FA3488234BA2831AC9C687AC6F8695A66F4D158D0C1D868F56743AF64F088164E38D68C83AF0CC5C4D6BA64989552EC6C07CAFCE149D3452FCB314180EC076600011019E4D66130F0B2A1A306083326A8C39B6C5789185E51BB62E4DFFD776679C16DCCBCFBAF08732286F9A7240092B0113EF75280276ECDCC1594DED45277AB49C7069AE338E7BAEEEC7FF7E52B664F40B8327422AF41C808B0F4E36AF65FA17EDFD38B4FD26944B21902C1EF767EC1F8FED434F4F58A1C52566D4C7C9AD4F0AE6F6E64EC5F4D15ED266AADCC931E75FD717D3EE70E10EC43BFEA9DE8566A1DC7CE550EEC7B33753CBC4204E2CEDC3DF455A3E7B79BBF037E5ADF1FD48AD12F2B4D7BA8D013CBA05935DAA7128E4723DEBF28BA86966A42772622B81C8CAB176354932AF1BAFC1D10297D22DE362A'
    context = ''
    addrnd = '77D8D46C1A12D5B0C2C3C6825A8D44ADB1706A6F9C498F7E'
    # Expected hash of signature
    expected_hash = 'f2912e9f545cb9c3b5a40166edd7ba2f5141a7bfca9174c6a41736e4774fc5fd'

    sighex, sigannotated = slhdsa_sign(params, msg, sk, addrnd)
    save_sig_to_file(params.name, sighex, sigannotated, TESTDIR)
    print(f"HASH(ok) ={expected_hash}")
    print(headtail(sighex, 48*3))


def main():
    # COMMENT OUT CASES AS REQUIRED
    # NB the 256 cases take some time, especially 256s

    # Ensure TESTDIR exists
    if not os.path.exists(TESTDIR):
        os.makedirs(TESTDIR, exist_ok=True)
        print("Created directory", TESTDIR)
    #show_params()
    # 12 parameter sets...
    case_sha2_128f_1()
    #case_sha2_128s_163()
    #case_sha2_192f_50()
    #case_sha2_192s_280()
    #case_sha2_256s_513()
    #case_sha2_256f_353()
    #case_shake_128f_62()
    #case_shake_128s_219()
    #case_shake_192f_395()
    #case_shake_192s_546()
    #case_shake_256f_99()
    #case_shake_256s_564()

    print("\nALL DONE.")


if __name__ == '__main__':
    main()
